/*
 * Written by Dawid Kurzyniec and released to the public domain, as explained
 * at http://creativecommons.org/licenses/publicdomain
 */

package edu.emory.mathcs.util.net.ssl;

import java.io.*;
import java.net.*;
import java.nio.channels.*;
import javax.net.ssl.*;
import edu.emory.mathcs.util.net.*;
import java.util.zip.*;

/**
 * Wrapper for SSL server sockets which enables to add functionality in
 * subclasses on top of existing, bound SSL server sockets. It is useful when
 * direct subclassing of delegate socket class is not possible, e.g. if the
 * delegate socket is created by a library. Possible usage example is socket
 * factory chaining. This class delegates all socket-related requests to the
 * wrapped delegate, as of JDK 1.4.
 *
 * @author Dawid Kurzyniec
 * @version 1.4
 */
public class SSLServerSocketWrapper extends SSLServerSocket {

    /**
     * the wrapped delegate socket.
     */
    final SSLServerSocket delegate;

    /**
     * Creates new socket wrapper for a given server socket. The delegate
     * must be bound and it must not be closed.
     * @param delegate the delegate socket to wrap.
     * @throws SocketException if the delegate socket is closed, not bound,
     *                         or not connected.
     */
    protected SSLServerSocketWrapper(SSLServerSocket delegate) throws IOException {
        if (delegate == null) {
            throw new NullPointerException();
        }
        if (delegate.isClosed()) {
            throw new SocketException("Delegate server socket is closed");
        }
        if (!(delegate.isBound())) {
            throw new SocketException("Delegate server socket is not bound");
        }
        this.delegate = delegate;
    }

    protected SSLSocket wrapAcceptedSocket(SSLSocket accepted) throws SocketException {
        return new SSLSocketWrapper(accepted);
    }

    public boolean isBound() {
        return true;
    }

    public boolean isClosed() {
        return delegate.isClosed();
    }

    // no need to overwrite bind() methods - the socket is always bound,
    // so they always throw SocketException

    public InetAddress getInetAddress() {
        return delegate.getInetAddress();
    }

    public int getLocalPort() {
        return delegate.getLocalPort();
    }

    public SocketAddress getLocalSocketAddress() {
        return delegate.getLocalSocketAddress();
    }

    public Socket accept() throws IOException {
        SSLSocket socket = (SSLSocket)delegate.accept();
        return wrapAcceptedSocket(socket);
    }

    public void close() throws IOException {
        delegate.close();
    }

    public ServerSocketChannel getChannel() {
        return delegate.getChannel();
    }

    public void setSoTimeout(int timeout) throws SocketException {
        delegate.setSoTimeout(timeout);
    }

    public int getSoTimeout() throws IOException {
        return delegate.getSoTimeout();
    }

    public void setReuseAddress(boolean on) throws SocketException {
        delegate.setReuseAddress(on);
    }

    public boolean getReuseAddress() throws SocketException {
        return delegate.getReuseAddress();
    }

    public String toString() {
        return delegate.toString();
    }

    public void setReceiveBufferSize (int size) throws SocketException {
        delegate.setReceiveBufferSize(size);
    }

    public int getReceiveBufferSize() throws SocketException{
        return delegate.getReceiveBufferSize();
    }


    public String[] getSupportedProtocols() {
        return delegate.getSupportedProtocols();
    }

    public String[] getEnabledProtocols() {
        return delegate.getEnabledProtocols();
    }

    public void setEnabledProtocols(String[] protocols) {
        delegate.setEnabledProtocols(protocols);
    }

    public String[] getSupportedCipherSuites() {
        return delegate.getSupportedCipherSuites();
    }

    public String[] getEnabledCipherSuites() {
        return delegate.getEnabledCipherSuites();
    }

    public void setEnabledCipherSuites(String[] enabledCipherSuites) {
        delegate.setEnabledCipherSuites(enabledCipherSuites);
    }

    public boolean getWantClientAuth() {
        return delegate.getWantClientAuth();
    }

    public void setWantClientAuth(boolean want) {
        delegate.setWantClientAuth(want);
    }

    public boolean getNeedClientAuth() {
        return delegate.getNeedClientAuth();
    }

    public void setNeedClientAuth(boolean want) {
        delegate.setNeedClientAuth(want);
    }

    public boolean getUseClientMode() {
        return delegate.getUseClientMode();
    }

    public void setUseClientMode(boolean clientMode) {
        delegate.setUseClientMode(clientMode);
    }

    public boolean getEnableSessionCreation() {
        return delegate.getEnableSessionCreation();
    }

    public void setEnableSessionCreation(boolean enable) {
        delegate.setEnableSessionCreation(enable);
    }


    public boolean equals(Object obj) {
        if (!(obj instanceof SSLServerSocketWrapper)) return false;
        SSLServerSocketWrapper that = (SSLServerSocketWrapper)obj;
        return this.delegate.equals(that.delegate);
    }

    public int hashCode() {
        return delegate.hashCode() ^ 0x01010101;
    }

}
